<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Knowledgebase\Library\Article;

use Knowledgebase\Admin\LoaderMock;
use SWIFT;
use SWIFT_Exception;

/**
 * Class KnowledgebaseArticleManagerTest
 */
#[\PHPUnit\Framework\Attributes\Group('knowledgebase')]
class KnowledgebaseArticleManagerTest extends \SWIFT_TestCase
{
    public static $_next = 0;
    private ?\Knowledgebase\Library\Article\SWIFT_KnowledgebaseArticleManagerMock $obj;
    public static $_record = [
        'articlestatus' => 1,
        'kbarticleid' => 1,
        'kbcategoryid' => 1,
        'linktypeid' => 1,
        'allowcomments' => 0,
        'allowrating' => 0,
        'uservisibilitycustom' => 1,
        'staffvisibilitycustom' => 1,
        'categorytype' => 1,
        'parentkbcategoryid' => 1,
    ];

    /**
     * @throws SWIFT_Exception
     */
    public function setUp(): void
    {
        parent::setUp();

        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('NextRecord')->willReturnCallback(function () {
            self::$_next++;

            if (self::$_next === 7 || self::$_next === 19) {
                self::$_record['linktypeid'] = '0';
            } else {
                self::$_record['linktypeid'] = '1';
            }

            if (self::$_next === 17 || self::$_next === 29) {
                self::$_record['categorytype'] = 4;
            } else {
                self::$_record['categorytype'] = 1;
            }

            return in_array(self::$_next, [1, 2, 6, 7, 9, 11, 13, 15, 17, 19, 21, 23, 25, 27, 29], true);
        });

        $mockDb->method('QueryFetch')->willReturnCallback(function ($x) {
            if (str_contains($x, "kbarticleid = '2'")) {
                return false;
            }

            $arr = [
                'kbarticleid' => 1,
                'articlestatus' => 1,
                'allowcomments' => 0,
                'allowrating' => 0,
                'uservisibilitycustom' => 1,
                'staffvisibilitycustom' => 1,
                'categorytype' => 1,
                'kbcategoryid' => 1,
            ];

            if (str_contains($x, "kbarticleid = '3'")) {
                $arr['articlestatus'] = 2;
            }

            if (str_contains($x, "kbcategoryid = '2'")) {
                $arr['kbcategoryid'] = 2;
            }

            return $arr;
        });

        $mockDb->Record = &self::$_record;

        SWIFT::GetInstance()->Database = $mockDb;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturn([
            '7' => [1 => [1]],
            '5' => [1 => [1]],
        ]);

        SWIFT::GetInstance()->Cache = $mockCache;

        SWIFT::GetInstance()->Load = new LoaderMock();

        $mockStaff = $this->createMock(\Base\Models\Staff\SWIFT_Staff::class);

        $mockStaff->method('GetProperty')->willReturn('1');

        SWIFT::GetInstance()->Staff = $mockStaff;

        $this->obj = new SWIFT_KnowledgebaseArticleManagerMock([]);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testRetrieveBySeoSubjectReturnsFalse()
    {
        $this->assertFalse($this->obj->RetrieveBySeoSubject('subject'),
            'Returns false on empty results');
        $this->assertFalse($this->obj->RetrieveBySeoSubject('subject'),
            'Returns false on second run');
    }

    /**
     * @throws SWIFT_Exception
     */
    #[\PHPUnit\Framework\Attributes\Depends('testRetrieveBySeoSubjectReturnsFalse')]
    public function testRetrieveForUserReturnsArray()
    {
        $this->assertFalse($this->obj->RetrieveForUser(2),
            'Returns false if article not found');
        $this->assertFalse($this->obj->RetrieveForUser(3),
            'Returns false if articlestatus != published');
        $this->assertIsArray($this->obj->RetrieveForUser(4, 2),
            'Returns array');
        $this->assertIsArray($this->obj->RetrieveForUser(1, 3),
            'Returns array with invalid categoryid');
        $this->assertIsArray($this->obj->RetrieveForUser(1, 3),
            'Returns array with categorytype = inherit');
    }

    /**
     * @throws SWIFT_Exception
     */
    #[\PHPUnit\Framework\Attributes\Depends('testRetrieveForUserReturnsArray')]
    public function testRetrieveForStaffReturnsArray()
    {
        $this->assertFalse($this->obj->RetrieveForStaff(2),
            'Returns false if article not found');
        $this->assertFalse($this->obj->RetrieveForStaff(3),
            'Returns false if articlestatus != published');
        $this->assertIsArray($this->obj->RetrieveForStaff(4, 2),
            'Returns array');
        $this->assertIsArray($this->obj->RetrieveForStaff(1, 3),
            'Returns array with invalid categoryid');
        $this->assertIsArray($this->obj->RetrieveForStaff(1, 3),
            'Returns array with categorytype = inherit');
        $this->assertFalse($this->obj->RetrieveForStaff(1),
            'Returns false without categories');
    }
}

class SWIFT_KnowledgebaseArticleManagerMock extends SWIFT_KnowledgebaseArticleManager
{
    /**
     * SWIFT_KnowledgebaseArticleManagerMock constructor.
     * @param array $services
     * @throws SWIFT_Exception
     */
    public function __construct(array $services = [])
    {
        $this->Load = new LoaderMock();
        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }
        parent::__construct();
    }

    public function Initialize()
    {
        return true;
    }
}
