<?php
/**
* ###############################################
*
* Kayako Classic
* _______________________________________________
*
* @author        Werner Garcia <werner.garcia@crossover.com>
*
* @package       swift
* @copyright     Copyright (c) 2001-2018, Trilogy
* @license       http://kayako.com/license
* @link          http://kayako.com
*
* ###############################################
*/

namespace Knowledgebase\Library\Render;

use Knowledgebase\Admin\LoaderMock;
use SWIFT;
use SWIFT_Exception;

/**
* Class KnowledgebaseRenderManagerTest
*/
#[\PHPUnit\Framework\Attributes\Group('knowledgebase')]
class KnowledgebaseRenderManagerTest extends \SWIFT_TestCase
{
    public static $_count = [];

    /**
     * @return SWIFT_KnowledgebaseRenderManagerMock
     * @throws SWIFT_Exception
     */
    public function getLibrary()
    {
        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);

        $this->mockProperty($mockDb, 'Record', [
            'kbarticleid' => 1,
            'totalitems' => 1,
            'kbcategoryid' => 1,
        ]);

        SWIFT::GetInstance()->Database = $mockDb;

        $mockStaff = $this->createMock(\Base\Models\Staff\SWIFT_Staff::class);

        $mockStaff->method('GetProperty')->willReturn(1);

        SWIFT::GetInstance()->Staff = $mockStaff;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturn([]);

        SWIFT::GetInstance()->Cache = $mockCache;

        SWIFT::GetInstance()->Load = new LoaderMock();

        $mockLang = $this->createMock('SWIFT_LanguageEngine');

        $mockLang->method('Get')->willReturnCallback(function ($x) {
            if ($x === 'charset') {
                return 'UTF-8';
            }

            return $x;
        });
        \SWIFT::GetInstance()->Language = $mockLang;

        $mockSettings = $this->createMock('SWIFT_Settings');

        $mockSettings->method('Get')->willReturn('1');

        SWIFT::GetInstance()->Settings = $mockSettings;

        $mockTpl = $this->createMock('SWIFT_TemplateEngine');

        return new SWIFT_KnowledgebaseRenderManagerMock([
            'Cache' => SWIFT::GetInstance()->Cache,
            'Language' => $mockLang,
            'Settings' => $mockSettings,
            'Database' => $mockDb,
            'Template' => $mockTpl,
        ]);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getLibrary();
        $this->assertInstanceOf(\Knowledgebase\Library\Render\SWIFT_KnowledgebaseRenderManager::class, $obj);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testRenderTreeReturnsHtml()
    {
        $obj = $this->getLibrary();
        $this->assertStringContainsString('<ul class="swifttree">', $obj->RenderTree());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->RenderTree();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testRenderViewKnowledgebaseTreeReturnsHtml()
    {
        $obj = $this->getLibrary();
        $this->assertStringContainsString('<ul class="swifttree">', $obj->RenderViewKnowledgebaseTree());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->RenderViewKnowledgebaseTree();
    }

    public function testRenderTreeOrder()
    {
        $services = $this->getMockServices();
        $services['Database']->method('Query')
            ->will($this->returnCallback(
                function($query) {
                    $this->assertTrue(1 === preg_match('/.*ORDER BY displayorder ASC$/', $query));
                }
            ));

        $render = new SWIFT_KnowledgebaseRenderManager();
        $render->RenderTree();
    }
}

class SWIFT_KnowledgebaseRenderManagerMock extends SWIFT_KnowledgebaseRenderManager
{
    /**
     * SWIFT_KnowledgebaseRenderManagerMock constructor.
     * @param array $services
     * @throws SWIFT_Exception
     */
    public function __construct(array $services = [])
    {
        $this->Load = new LoaderMock();
        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }
        parent::__construct();
    }

    public function Initialize()
    {
        return true;
    }
}
