<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Knowledgebase\Staff;

use Knowledgebase\Admin\LoaderMock;
use Knowledgebase\Models\Article\SWIFT_KnowledgebaseArticle;
use SWIFT;
use SWIFT_Exception;

/**
 * Class Controller_ViewKnowledgebaseTest
 */
#[\PHPUnit\Framework\Attributes\Group('knowledgebase')]
class Controller_ViewKnowledgebaseTest extends \SWIFT_TestCase
{
    public function setUp(): void
    {
        parent::setUp();

        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('AutoExecute')->willReturn(true);
        $mockDb->method('Insert_ID')->willReturn(1);
        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);
        $mockDb->method('QueryFetch')->willReturnCallback(function ($x) {
            if (str_contains($x, "kbarticleid = '0'")) {
                return [
                    'kbarticleid' => 0,
                ];
            }

            if (str_contains($x, "kbarticleid = '2'")) {
                return [
                    'kbarticleid' => 2,
                    'newstype' => '3',
                    'newsstatus' => '0',
                ];
            }

            if (str_contains($x, "attachmentid = '2'")) {
                return [
                    'attachmentid' => 2,
                    'linktype' => 2,
                    'linktypeid' => 2,
                ];
            }

            return [
                'newsitemid' => 1,
                'kbarticleid' => 1,
                'attachmentid' => 1,
                'linktype' => 5,
                'linktypeid' => 1,
                'newstype' => '1',
                'newsstatus' => '2',
                'allowcomments' => '1',
                'uservisibilitycustom' => '1',
                'staffvisibilitycustom' => '2',
            ];
        });

        $this->mockProperty($mockDb, 'Record', [
            'kbarticleid' => '1',
        ]);

        SWIFT::GetInstance()->Database = $mockDb;

        $mockStaff = $this->createMock(\Base\Models\Staff\SWIFT_Staff::class);

        $mockStaff->method('GetProperty')->willReturnArgument(0);
        $mockStaff->method('GetStaffID')->willReturn(1);
        $mockStaff->method('GetIsClassLoaded')->willReturn(true);
        $mockStaff->method('GetPermission')
            ->willReturnOnConsecutiveCalls('0', '1');

        SWIFT::GetInstance()->Staff = $mockStaff;

        $mockSession = $this->createMock('SWIFT_Session');

        $mockSession->method('GetIsClassLoaded')->willReturn(true);
        $mockSession->method('GetProperty')->willReturnArgument(0);

        SWIFT::GetInstance()->Session = $mockSession;

        $mockRouter = $this->createMock('SWIFT_Router');

        SWIFT::GetInstance()->Router = $mockRouter;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturnOnConsecutiveCalls([], [
            [
                'appname' => 'knowledgebase',
                'isenabled' => '1',
            ],
        ]);

        SWIFT::GetInstance()->Cache = $mockCache;
    }

    /**
     * @return Controller_ViewKnowledgebaseMock
     * @throws SWIFT_Exception
     */
    public function getController(array $services = [])
    {
        $mockLang = $this->createMock('SWIFT_LanguageEngine');

        $mockLang->method('Get')->willReturnArgument(0);

        $mockView = $this->createMock(\Knowledgebase\Staff\View_ViewKnowledgebase::class);

        $mockInt = $this->createMock(\Base\Library\UserInterface\SWIFT_UserInterfaceControlPanel::class);

        $mockTpl = $this->createMock('SWIFT_TemplateEngine');

        $mockKb = $this->createMock(\Knowledgebase\Library\Article\SWIFT_KnowledgebaseArticleManager::class);
        $mockKb->method('RetrieveForStaff')->willReturnCallback(function ($x) {
            if ($x === 0) {
                return false;
            }
            if ($x === 2) {
                return [0];
            }
            return [
                new SWIFT_KnowledgebaseArticle(new \SWIFT_DataStore([
                    'kbarticleid' => 1,
                    'views' => 1,
                    'subject' => 'subject',
                    'ratinghits' => 1,
                    'ratingcount' => 1,
                ])),
            ];
        });

        $mockMgr = $this->createMock(\Base\Library\Comment\SWIFT_CommentManager::class);

        $mockMgr->method('ProcessPOSTUser')->willReturn(true);

        $mockSettings = $this->createMock('SWIFT_Settings');

        $mockSettings->method('Get')->willReturn(1);

        $mockRender = $this->createMock(\Knowledgebase\Library\Render\SWIFT_KnowledgebaseRenderManager::class);
        $mockRender->method('RenderViewKnowledgebaseTree')->willReturn('');

        return new Controller_ViewKnowledgebaseMock(array_merge([
            'Template' => $mockTpl,
            'Language' => $mockLang,
            'UserInterface' => $mockInt,
            'View' => $mockView,
            'CommentManager' => $mockMgr,
            'Settings' => $mockSettings,
            'KnowledgebaseArticleManager' => $mockKb,
            'KnowledgebaseRenderManager' => $mockRender,
        ], $services));
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = new Controller_ViewKnowledgebase();
        $this->assertInstanceOf(\Knowledgebase\Staff\Controller_ViewKnowledgebase::class, $obj);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testLoadDisplayDataForViewArticleReturnsTrue()
    {
        $obj = $this->getController();
        $this->assertTrue($obj->_LoadDisplayDataForViewArticle());
        $this->expectException('SWIFT_Exception');
        $obj->SetIsClassLoaded(false);
        $obj->_LoadDisplayDataForViewArticle();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testIndexReturnsTrue()
    {
        $obj = $this->getController();
        $this->assertTrue($obj->Index(),
            'Returns true with staff_kbcanviewarticles = 0');
        $this->assertTrue($obj->Index(),
            'Returns true with staff_kbcanviewarticles = 1');
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->SetIsClassLoaded(false);
        $obj->Index();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testArticleReturnsTrue()
    {
        $obj = $this->getController();
        $this->assertFalse($obj->Article(0),
            'Returns false with invalid id');
        $this->assertTrue($obj->Article(1),
            'Returns true with staff_kbcanviewarticles = 0');
        $this->assertTrue($obj->Article(1),
            'Returns true with staff_kbcanviewarticles = 1');
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->SetIsClassLoaded(false);
        $obj->Article(1);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testProcessArticleRatingReturnsArray()
    {
        $art = new SWIFT_KnowledgebaseArticle(new \SWIFT_DataStore([
            'kbarticleid' => 1,
            'articlerating' => 1,
        ]));
        $arr = ['_hasNotRated' => false, '_articleRating' => 1];
        $cookieMock = $this->createMock('SWIFT_Cookie');
        $cookieMock->method('GetVariable')->willReturn(true);
        $obj = $this->getController([
            'Cookie' => $cookieMock,
        ]);
        $this->assertEquals($arr, $obj->_ProcessArticleRating($art));
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->SetIsClassLoaded(false);
        $obj->_ProcessArticleRating($art);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testRateThrowsException()
    {
        $obj = $this->getController();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj->Rate(2);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testRateReturnsTrue()
    {
        $cookieMock = $this->createMock('SWIFT_Cookie');

        $obj = $this->getController([
            'Cookie' => $cookieMock,
        ]);

        $this->assertFalse($obj->Rate(0),
            'Returns false with invalid id');

        $this->assertTrue($obj->Rate(1, 0),
            'Returns true with _isHelpful = 0');

        $this->assertTrue($obj->Rate(1, 1),
            'Returns true with _isHelpful = 1');
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->SetIsClassLoaded(false);
        $obj->Rate(1);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetAttachmentThrowsException()
    {
        $obj = $this->getController();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj->GetAttachment(2, 1);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetAttachmentThrowsInvalidDataException()
    {
        $obj = $this->getController();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj->GetAttachment(1, 2);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetAttachmentReturnsTrue()
    {
        $obj = $this->getController();

        $this->assertFalse($obj->GetAttachment(0, 1),
            'Returns false with invalid id');

        $this->assertTrue($obj->GetAttachment(1, 1),
            'Returns true');
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->SetIsClassLoaded(false);
        $obj->GetAttachment(1, 1);
    }
}

class Controller_ViewKnowledgebaseMock extends Controller_ViewKnowledgebase
{
    /**
     * Controller_ViewKnowledgebaseMock constructor.
     * @param array $services
     * @throws SWIFT_Exception
     */
    public function __construct(array $services = [])
    {
        $this->Load = new LoaderMock();

        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }

        parent::__construct();
    }

    public function Initialize()
    {
        return true;
    }
}
