<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace News\Models\Category;

use News\Admin\LoaderMock;
use SWIFT;

/**
 * Class SWIFT_NewsCategoryLinkTest
 */
#[\PHPUnit\Framework\Attributes\Group('news')]
class SWIFT_NewsCategoryLinkTest extends \SWIFT_TestCase
{
    /**
     * @param int $_newsCategoryLinkID
     * @return SWIFT_NewsCategoryLinkMock
     * @throws SWIFT_Category_Exception
     */
    public function getModel($_newsCategoryLinkID = 1)
    {
        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);
        $mockDb->method('QueryFetch')->willReturnCallback(function ($x) {
            if (str_contains($x, "newscategorylinkid = '0'")) {
                return false;
            }

            return [
                'newscategorylinkid' => 1,
            ];
        });

        SWIFT::GetInstance()->Database = $mockDb;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturn([]);

        SWIFT::GetInstance()->Cache = $mockCache;

        SWIFT::GetInstance()->Load = new LoaderMock();

        $mockLang = $this->createMock('SWIFT_LanguageEngine');

        $mockLang->method('Get')->willReturnArgument(0);

        $mockSettings = $this->createMock('SWIFT_Settings');

        $mockSettings->method('Get')->willReturnOnConsecutiveCalls('1', '0');

        return new SWIFT_NewsCategoryLinkMock($_newsCategoryLinkID, [
            'Language' => $mockLang,
            'Settings' => $mockSettings,
            'Database' => SWIFT::GetInstance()->Database,
        ]);
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testConstructorThrowsException()
    {
        $this->expectExceptionMessage('Failed to load News Category Link ID: 0');
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $this->getModel(0);
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getModel();
        $this->assertInstanceOf(\News\Models\Category\SWIFT_NewsCategoryLink::class, $obj);

        $obj->__destruct();
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testProcessUpdatePoolReturnsTrue()
    {
        $obj = $this->getModel();
        $obj->UpdatePool('key', 'val');
        $this->assertTrue($obj->ProcessUpdatePool(),
            'Returns true after updating');

        $obj->SetIsClassLoaded(false);
        $this->assertFalse($obj->ProcessUpdatePool(),
            'Returns false if class is not loaded');
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testGetNewsCategoryLinkIDThrowsException()
    {
        $obj = $this->getModel();
        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj->GetNewsCategoryLinkID();
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testGetDataStoreReturnsArray()
    {
        $obj = $this->getModel();
        $this->assertIsArray($obj->GetDataStore());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj->GetDataStore();
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testGetPropertyThrowsException()
    {
        $obj = $this->getModel();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj->GetProperty('invalid');
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testGetPropertyReturnsValue()
    {
        $obj = $this->getModel();
        $this->assertEquals('1', $obj->GetProperty('newscategorylinkid'));

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj->GetProperty('prop');
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testCreateThrowsInvalidDataException()
    {
        $obj = $this->getModel();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj::Create(0, 0);
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testCreateThrowsCreateFailedException()
    {
        $obj = $this->getModel();
        $this->expectExceptionMessage(SWIFT_CREATEFAILED);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj::Create(1, 1);
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testDeleteReturnsTrue()
    {
        $obj = $this->getModel();
        $this->assertTrue($obj->Delete(),
            'Returns true after deleting');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException(\News\Models\Category\SWIFT_Category_Exception::class);
        $obj->Delete();
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testDeleteListReturnsFalse()
    {
        $obj = $this->getModel();
        $this->assertFalse($obj::DeleteList([]),
            'Returns false if empty array');

        SWIFT::GetInstance()->Database->NextRecord(); //advance pointer
        $this->assertFalse($obj::DeleteList([0]),
            'Returns false with invalid array');
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testDeleteOnNewsCategoryReturnsFalse()
    {
        $obj = $this->getModel();
        $this->assertFalse($obj::DeleteOnNewsCategory([]),
            'Returns false if empty array');

        SWIFT::GetInstance()->Database->NextRecord(); //advance pointer
        $this->assertFalse($obj::DeleteOnNewsCategory([0]),
            'Returns false with invalid array');
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testDeleteOnNewsItemReturnsFalse()
    {
        $obj = $this->getModel();
        $this->assertFalse($obj::DeleteOnNewsItem([]),
            'Returns false if empty array');
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testRetrieveOnNewsItemReturnsEmptyArray()
    {
        $obj = $this->getModel();
        $this->assertEmpty($obj::RetrieveOnNewsItem([]));
    }

    /**
     * @throws SWIFT_Category_Exception
     */
    public function testRetrieveOnNewsCategoryReturnsEmptyArray()
    {
        $obj = $this->getModel();
        $this->assertEmpty($obj::RetrieveOnNewsCategory([]));
    }
}

class SWIFT_NewsCategoryLinkMock extends SWIFT_NewsCategoryLink
{
    /**
     * SWIFT_NewsCategoryLinkMock constructor.
     * @param $_newsCategoryLinkID
     * @param array $services
     * @throws SWIFT_Category_Exception
     */
    public function __construct($_newsCategoryLinkID, array $services = [])
    {
        $this->Load = new LoaderMock();
        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }
        parent::__construct($_newsCategoryLinkID);
    }

    public function Initialize()
    {
        return true;
    }
}
