<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Banjo Mofesola Paul <banjo.paul@aurea.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Parser\Library\Loop;

use Knowledgebase\Admin\LoaderMock;
use PHPUnit\Framework\Constraint\IsType;
use SWIFT_Exception;

/**
 * Class LoopCheckerTest
 */
#[\PHPUnit\Framework\Attributes\Group('parser-library')]
class LoopCheckerTest extends \SWIFT_TestCase
{
    /**
     * @throws SWIFT_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getMocked();
        $this->assertInstanceOf(\Parser\Library\Loop\SWIFT_LoopChecker::class, $obj);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testCheckCompletely()
    {
        $obj = $this->getMocked();

        $this->assertFalse($obj->Check('email@address.com'));

        // make the email blocked, for code coverage
        \SWIFT::GetInstance()->Database->method('QueryFetch')->willReturn([
            'totalitems' => 1
        ]);
        $this->assertTrue($obj->Check('email@address.com'));

        $obj->SetIsClassLoaded(false);
        $this->assertClassNotLoaded($obj, 'Check', 'email@address.com');
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testCheckEmailForLoopThrowsException()
    {
        $obj = $this->getMocked();

        $obj->SetIsClassLoaded(false);
        $this->assertClassNotLoaded($obj, 'CheckEmailForLoop', 'email');
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testCheckForEmailLoopWorkerReturnsFalse()
    {
        $obj = $this->getMocked();

        $this->assertFalse($obj->CheckForEmailLoopWorker($obj->_compareTime, $obj->_datelineHitList, $obj->_loopRuleContainer));

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException(SWIFT_Exception::class);
        $obj->CheckForEmailLoopWorker($obj->_compareTime, $obj->_datelineHitList, $obj->_loopRuleContainer);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testCheckForEmailLoopWorkerReturnsString()
    {
        $obj = $this->getMocked();

        $obj->_datelineHitList[] = 1;
        $obj->_loopRuleContainer[] = [ 'length' => 1, 'maxhits' => 0, 'restoreafter' => 'restoreafter' ];

        $this->assertEquals('restoreafter', $obj->CheckForEmailLoopWorker($obj->_compareTime,
            $obj->_datelineHitList, $obj->_loopRuleContainer));
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testLoopException(): never
    {
        $obj = $this->getMocked();

        $this->expectException(SWIFT_Loop_Exception::class);
        throw new SWIFT_Loop_Exception('message');
    }

    private function getMocked(): \PHPUnit\Framework\MockObject\MockObject|\Parser\Library\Loop\SWIFT_LoopCheckerMock
    {
        return $this->getMockObject(\Parser\Library\Loop\SWIFT_LoopCheckerMock::class);
    }
}

class SWIFT_LoopCheckerMock extends SWIFT_LoopChecker
{
    public $_compareTime;
    public $_datelineHitList;
    public $_loopRuleContainer;

    public function __construct($services = [])
    {
        $this->Load = new LoaderMock();

        foreach ($services as $key => $service) {
            $this->$key = $service;
        }

        $this->SetIsClassLoaded(true);

        parent::__construct();
    }

    public function CheckEmailForLoop($_emailAddress)
    {
        return parent::CheckEmailForLoop($_emailAddress); // TODO: Change the autogenerated stub
    }

    public function CheckForEmailLoopWorker($_compareTime, &$_datelineHitList, &$_loopRuleContainer)
    {
        return parent::CheckForEmailLoopWorker($_compareTime, $_datelineHitList, $_loopRuleContainer); // TODO: Change the autogenerated stub
    }

    public function Initialize()
    {
        // override
        $this->_compareTime = 1;
        $this->_datelineHitList = [ ];
        $this->_loopRuleContainer = [ ];
        return true;
    }
}

