<?php
/**
* ###############################################
*
* Kayako Classic
* _______________________________________________
*
* @author        Werner Garcia <werner.garcia@crossover.com>
*
* @package       swift
* @copyright     Copyright (c) 2001-2018, Trilogy
* @license       http://kayako.com/license
* @link          http://kayako.com
*
* ###############################################
*/

namespace Troubleshooter\Api;

use Troubleshooter\Admin\LoaderMock;

/**
* Class ApiController_CommentTest
*/
#[\PHPUnit\Framework\Attributes\Group('troubleshooter')]
class ApiController_CommentTest extends \SWIFT_TestCase
{
    /**
     * @return Controller_CommentMock
     * @throws \SWIFT_Exception
     */
    protected function getController(array $services = [])
    {
        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('Insert_ID')
            ->willReturn(1);

        $mockDb->method('QueryFetch')
            ->willReturnCallback(function ($x) {
                if (str_contains($x, "commentid = '0'") ||
                    str_contains($x, "troubleshooterstepid = '2'") ||
                    str_contains($x, "userid = '2'")) {
                    return false;
                }

                return [
                    'troubleshooterstepid' => 1,
                    'userid' => 1,
                    'staffid' => 1,
                    'troubleshootercategoryid' => 1,
                    'commentid' => 1,
                    'typeid' => 1,
                    'creatortype' => 1,
                    'creatorid' => 1,
                    'fullname' => 'full name',
                    'email' => 'user@mail.com',
                    'ipaddress' => 'ipaddress',
                    'dateline' => time(),
                    'parentcommentid' => 1,
                    'commentstatus' => 'commentstatus',
                    'useragent' => 'useragent',
                    'referrer' => 'referrer',
                    'parenturl' => 'parenturl',
                    'contents' => 'contents',
                ];
            });

        $mockDb->method('NextRecord')
            ->willReturnOnConsecutiveCalls(true, false);

        $this->mockProperty($mockDb, 'Record', [
            'commentid' => 1,
            'typeid' => 1,
            'creatortype' => 1,
            'creatorid' => 1,
            'fullname' => 'full name',
            'email' => 'user@mail.com',
            'ipaddress' => 'ipaddress',
            'dateline' => time(),
            'parentcommentid' => 1,
            'commentstatus' => 'commentstatus',
            'useragent' => 'useragent',
            'referrer' => 'referrer',
            'parenturl' => 'parenturl',
            'contents' => 'contents',
        ]);

        \SWIFT::GetInstance()->Database = $mockDb;

        $settings = $this->createMock('SWIFT_Settings');
        $settings->method('Get')
            ->willReturn('1');

        $mgr = $this->createMock('SWIFT_RESTManager');
        $mgr->method('Authenticate')
            ->willReturn(true);

        $svr = $this->createMock('SWIFT_RESTServer');
        $svr->method('GetVariableContainer')
            ->willReturn(['salt' => '1']);

        return new Controller_CommentMock($settings, $mgr, $svr, $mockDb, $services);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getController();
        $this->assertInstanceOf(\Troubleshooter\Api\Controller_Comment::class, $obj);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testGetListPrintsXml() {
        $xml = $this->createMock('SWIFT_XML');

        $obj = $this->getController(['XML' => $xml]);
        $this->assertTrue($obj->GetList(),
            'Returns true after printing XML');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->GetList();
    }

    /**
     * @throws \ReflectionException
     * @throws \SWIFT_Exception
     */
    public function testProtectedMethodThrowsException() {
        $obj = $this->getController();
        $obj->SetIsClassLoaded(false);

        $reflectionClass = new \ReflectionClass($obj);
        $method = $reflectionClass->getMethod('ProcessTroubleshooterComments');
        $method->setAccessible(true);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $method->invoke($obj, false);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testGetPrintsXml() {
        $xml = $this->createMock('SWIFT_XML');

        $obj = $this->getController(['XML' => $xml]);
        $this->assertTrue($obj->Get(0),
            'Returns true after printing XML');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->Get(0);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testPutDoesNothing() {
        $obj = $this->getController();
        $this->expectOutputString('Put not implemented');
        $this->assertFalse($obj->Put(0),
            'Returns false because method is not implemented');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->Put(0);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testDeleteReturnsTrue() {
        $obj = $this->getController();
        $this->assertTrue($obj->Delete(0),
            'Returns true after deleting');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->Delete(0);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testPostPrintsXml(): never {
        $xml = $this->createMock('SWIFT_XML');

        // pass mock XML service
        $obj = $this->getController(['XML' => $xml]);

        unset($_POST['contents']);
        $this->assertFalse($obj->Post(),
            'Returns false without contents');

        $_POST['contents'] = 'contents';
        unset($_POST['troubleshooterstepid']);
        $this->assertFalse($obj->Post(),
            'Returns false without step');

        $_POST['troubleshooterstepid'] = 2;
        $this->assertFalse($obj->Post(),
            'Returns false without creator type');

        $_POST['creatortype'] = 3;
        $this->assertFalse($obj->Post(),
            'Returns false with non existent step');

        $_POST['troubleshooterstepid'] = 1;
        $_POST['creatorid'] = 0;
        $_POST['email'] = 'user@mail.com';
        $this->assertFalse($obj->Post(),
            'Returns false with invalid creator type');

        $_POST['creatortype'] = 1; // staff
        $this->assertFalse($obj->Post(),
            'Returns false with invalid creator id');

        $_POST['creatorid'] = 1;
        $_POST['parentcommentid'] = 1;
        $this->assertTrue($obj->Post(),
            'Returns true after printing XML');

        $_POST['creatortype'] = 2; // user

        $_POST['creatorid'] = 2;
        $this->assertFalse($obj->Post(),
            'Returns false with invalid creator id');

        $_POST['creatorid'] = 0;
        $this->assertFalse($obj->Post(),
            'Returns false with invalid creator id');

        $_POST['fullname'] = 'fullname';
        $this->assertTrue($obj->Post(),
            'Returns true after printing XML');

        $_POST['creatorid'] = 1;
        $this->assertTrue($obj->Post(),
            'Returns true after printing XML');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->Post();
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testListAllPrintsXml() {
        $xml = $this->createMock('SWIFT_XML');

        // pass mock XML service
        $obj = $this->getController(['XML' => $xml]);

        $this->assertTrue($obj->ListAll(),
            'Returns true after printing XML');

        $this->assertTrue($obj->ListAll(1),
            'Returns true after printing XML using id');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->ListAll();
    }
}

/**
 * Class Controller_CommentMock
 * @package Troubleshooter\Api
 */
class Controller_CommentMock extends Controller_Comment
{
    public function __construct($settings, public $RESTManager, public $RESTServer, $db, $services = [])
    {
        $this->Load = new LoaderMock();
        $this->Settings = $settings;
        $this->Database = $db;
        foreach ($services as $key => $service) {
            $this->$key = $service;
        }
        $this->SetIsClassLoaded(true);
        parent::__construct();
    }

    public function Initialize()
    {
        // override
        return true;
    }
}
