<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Tests\Api;

require_once dirname(__DIR__) . '/../__swift/config/config.php';

/**
 * Class BaseApiTestCase
 */
abstract class BaseApiTestCase extends \PHPUnit\Framework\TestCase
{
    protected $http;
    protected static $_url = '';
    protected static $_params = [];

    public function setUp(): void
    {
        if (empty(static::$_url)) {
            $_SWIFT = \SWIFT::GetInstance();
            $apiEnabled = $_SWIFT->Settings->GetBool('g_enableapiinterface');
            if (!$apiEnabled) {
                $this->fail('API interface is not enabled in settings');
                return;
            }
            static::$_url = $_SWIFT->Settings->Get('general_producturl');
            static::$_url .= ((str_ends_with((string) static::$_url, '/')) ? '' : '/') . 'api/index.php';
            $apiKey = $_SWIFT->Settings->GetKey('restapi', 'apikey');
            $secretKey = $_SWIFT->Settings->GetKey('restapi', 'secretkey');

            // We need to generate a random string of ten digits
            $salt = random_int(0, mt_getrandmax());

            // And then compute the signature by hashing the salt with the secret key as the key
            $signature = hash_hmac('sha256', $salt, (string) $secretKey, true);

            // Finally base64 encode...
            $encodedSignature = base64_encode($signature);

            static::$_params = [
                'apikey' => $apiKey,
                'salt' => $salt,
                'signature' => $encodedSignature,
            ];
        }

        $this->http = new \GuzzleHttp\Client();
    }

    public function tearDown(): void
    {
        $this->http = null;
    }

    /**
     * @param $endpoint
     * @param string $method
     * @param array $params
     * @param int $expectedStatusCode
     * @return mixed|\Psr\Http\Message\ResponseInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getResponse($endpoint, $method = 'GET', $params = [], $expectedStatusCode = 200)
    {
        if (in_array($method, ['POST', 'PUT'])) {
            $response = $this->http->request($method, static::$_url . '?e=' . $endpoint, [
                'form_params' => array_merge(static::$_params,
                    $params),
            ]);
        } else {
            $response = $this->http->request($method, static::$_url, [
                'query' => array_merge(static::$_params,
                    ['e' => $endpoint], $params),
            ]);
        }

        $this->assertEquals($expectedStatusCode, $response->getStatusCode());

        $contentType = $response->getHeaders()['Content-Type'][0];
        $this->assertStringContainsString('text/xml', $contentType);

        return $response;
    }

    /**
     * @param mixed|\Psr\Http\Message\ResponseInterface $response
     * @param int $options
     * @return mixed
     */
    public function getArrayFromResponse($response, $options = LIBXML_NOCDATA)
    {
        $str = $response->getBody()->getContents();
        $xml = simplexml_load_string((string) $str, 'SimpleXMLElement', $options);
        $json = json_encode($xml, JSON_THROW_ON_ERROR);
        return json_decode($json, true, 512, JSON_THROW_ON_ERROR);
    }

    /**
     * @param string $str
     * @param int $options
     * @return mixed
     */
    public function getArrayFromXml($str, $options = LIBXML_NOCDATA)
    {
        $xml = simplexml_load_string($str, 'SimpleXMLElement', $options);
        $json = json_encode($xml, JSON_THROW_ON_ERROR);
        return json_decode($json, true, 512, JSON_THROW_ON_ERROR);
    }
}
