<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Tests\Api\Knowledgebase;

use Base\Models\Comment\SWIFT_Comment;
use Base\Models\Staff\SWIFT_Staff;
use Base\Models\Staff\SWIFT_StaffGroup;
use Knowledgebase\Models\Article\SWIFT_KnowledgebaseArticle;
use Tests\Api\BaseApiTestCase;

/**
 * Class CommentTest
 */
#[\PHPUnit\Framework\Attributes\Group('knowledgebase')]
#[\PHPUnit\Framework\Attributes\Group('kbcomment')]
class CommentTest extends BaseApiTestCase
{
    private static $_id;
    private static ?int $_staffId = null;
    private static ?int $_staffGroupId = null;
    private static ?int $_articleId = null;

    public static function setUpBeforeClass(): void
    {
        try {
            \Colors::errlny('Creating StaffGroup "TestGroup"...');
            /** @var SWIFT_StaffGroup $_SWIFT_StaffGroupObject */
            $_SWIFT_StaffGroupObject = SWIFT_StaffGroup::Insert('TestGroup', false);
            static::$_staffGroupId = $_SWIFT_StaffGroupObject->GetID();
        } catch (\Exception $ex) {
            \Colors::errlnr('Unable to create StaffGroup: ' . $ex->getMessage());
        }

        try {
            \Colors::errlny('Creating Staff "Test Staff"...');
            /** @var SWIFT_Staff $_SWIFT_StaffObject */
            $_SWIFT_StaffObject = SWIFT_Staff::Create('Test', 'Staff', '',
                'teststaff' . static::$_staffGroupId, 'password', self::$_staffGroupId, 'teststaff@mail.com', '',
                '');
            static::$_staffId = $_SWIFT_StaffObject->GetID();
        } catch (\Exception $ex) {
            \Colors::errlnr('Unable to create Staff: ' . $ex->getMessage());
        }

        try {
            \Colors::errlny('Creating Article "Test Article"...');
            static::$_articleId = SWIFT_KnowledgebaseArticle::Create(SWIFT_KnowledgebaseArticle::CREATOR_STAFF,
                static::$_staffId, 'Test Staff', 'teststaff@mail.com', SWIFT_KnowledgebaseArticle::STATUS_PUBLISHED,
                'Test Subject', 'test-subject-seo', 'Test Contents');
        } catch (\Exception $ex) {
            \Colors::errlnr('Unable to create Article: ' . $ex->getMessage());
        }
    }

    public static function tearDownAfterClass(): void
    {
        if (isset(static::$_articleId)) {
            try {
                \Colors::errlny('Deleting Article "Test Article"...');
                SWIFT_KnowledgebaseArticle::DeleteList([static::$_articleId]);
            } catch (\Exception $ex) {
                \Colors::errlnr('Unable to delete Article: ' . $ex->getMessage());
            }
        }

        if (isset(static::$_staffId)) {
            try {
                \Colors::errlny('Deleting Staff "Test Staff"...');
                SWIFT_Staff::DeleteList([static::$_staffId]);
            } catch (\Exception $ex) {
                \Colors::errlnr('Unable to delete Staff: ' . $ex->getMessage());
            }
        }

        if (isset(static::$_staffGroupId)) {
            \Colors::errlny('Deleting StaffGroup "TestGroup"...');
            SWIFT_StaffGroup::DeleteList([static::$_staffGroupId]);
        }
    }

    /**
     * Test POST endpoint
     * ** THIS SHOULD BE THE FIRST TEST IN ORDER TO TEST CRUD **
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function testPost(): void
    {
        $response = $this->getResponse('/Knowledgebase/Comment', 'POST', [
            'knowledgebasearticleid' => static::$_articleId,
            'contents' => 'Test Contents',
            'creatortype' => SWIFT_Comment::CREATOR_STAFF,
            'creatorid' => static::$_staffId,
        ]);
        $list = $this->getArrayFromResponse($response);
        $this->assertEquals('Test Contents', $list['kbarticlecomment']['contents']);
        static::$_id = $list['kbarticlecomment']['id'];
    }

    /**
     * Test GET all endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testPost')]
    public function testGetAll(): void
    {
        $response = $this->getResponse('/Knowledgebase/Comment');
        $list = $this->getArrayFromResponse($response);
        $id = static::$_id;
        if (!isset($list['kbarticlecomment']['id'])) {
            $list = array_values(array_filter($list['kbarticlecomment'], fn($ug) => $ug['id'] === $id));
        } else {
            $list = [$list['kbarticlecomment']];
        }
        $this->assertEquals('Test Contents', $list[0]['contents']);
    }

    /**
     * Test GET all from Article ID endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testGetAll')]
    public function testGetAllFromArticleId(): void
    {
        $response = $this->getResponse('/Knowledgebase/Comment/ListAll/' . static::$_articleId);
        $list = $this->getArrayFromResponse($response);
        $this->assertEquals(static::$_id, $list['kbarticlecomment']['id']);
    }

    /**
     * Test GET from ID endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testGetAllFromArticleId')]
    public function testGetFromId(): void
    {
        $response = $this->getResponse('/Knowledgebase/Comment/' . static::$_id);
        $list = $this->getArrayFromResponse($response);
        $this->assertEquals('Test Contents', $list['kbarticlecomment']['contents']);
    }

    /**
     * Test DELETE endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testGetFromId')]
    public function testDelete(): void
    {
        $response = $this->getResponse('/Knowledgebase/Comment/' . static::$_id, 'DELETE');
        $this->assertEmpty($response->getBody()->getContents());
    }
}
