<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Tests\Api\Troubleshooter;

use Base\Models\Staff\SWIFT_Staff;
use Base\Models\Staff\SWIFT_StaffGroup;
use Tests\Api\BaseApiTestCase;
use Troubleshooter\Models\Category\SWIFT_TroubleshooterCategory;
use Troubleshooter\Models\Step\SWIFT_TroubleshooterStep;

/**
 * Class StepTest
 */
#[\PHPUnit\Framework\Attributes\Group('troubleshooter')]
#[\PHPUnit\Framework\Attributes\Group('step')]
class StepTest extends BaseApiTestCase
{
    private static $_id;
    private static ?int $_categoryId = null;
    private static ?int $_staffId = null;
    private static ?int $_staffGroupId = null;

    public static function setUpBeforeClass(): void
    {
        try {
            \Colors::errlny('Creating StaffGroup "TestGroup"...');
            /** @var SWIFT_StaffGroup $_SWIFT_StaffGroupObject */
            $_SWIFT_StaffGroupObject = SWIFT_StaffGroup::Insert('TestGroup', false);
            static::$_staffGroupId = $_SWIFT_StaffGroupObject->GetID();
        } catch (\Exception $ex) {
            \Colors::errlnr('Unable to create StaffGroup: ' . $ex->getMessage());
        }

        try {
            \Colors::errlny('Creating Staff "Test Staff"...');
            /** @var SWIFT_Staff $_SWIFT_StaffObject */
            $_SWIFT_StaffObject = SWIFT_Staff::Create('Test', 'Staff', '',
                'teststaff' . static::$_staffGroupId, 'password', self::$_staffGroupId, 'teststaff@mail.com', '',
                '');
            static::$_staffId = $_SWIFT_StaffObject->GetID();
        } catch (\Exception $ex) {
            \Colors::errlnr('Unable to create Staff: ' . $ex->getMessage());
        }

        try {
            \Colors::errlny('Creating Category "TestCategory"...');
            static::$_categoryId = SWIFT_TroubleshooterCategory::Create('TestCategory', 'TestCategory', SWIFT_TroubleshooterCategory::TYPE_GLOBAL, 0);
        } catch (\Exception $ex) {
            \Colors::errlnr('Unable to create Category: ' . $ex->getMessage());
        }
    }

    public static function tearDownAfterClass(): void
    {
        if (isset(static::$_id)) {
            try {
                SWIFT_TroubleshooterStep::DeleteList([static::$_id]); // cleanup
            } catch (\Exception $ex) {
            }
        }

        if (isset(static::$_categoryId)) {
            try {
            \Colors::errlny('Deleting Category "TestCategory"...');
            SWIFT_TroubleshooterCategory::DeleteList([static::$_categoryId]);
            } catch (\Exception $ex) {
                \Colors::errlnr('Unable to delete Category: ' . $ex->getMessage());
            }
        }

        if (isset(static::$_staffId)) {
            try {
                \Colors::errlny('Deleting Staff "Test Staff"...');
                SWIFT_Staff::DeleteList([static::$_staffId]);
            } catch (\Exception $ex) {
                \Colors::errlnr('Unable to delete Staff: ' . $ex->getMessage());
            }
        }

        if (isset(static::$_staffGroupId)) {
            \Colors::errlny('Deleting StaffGroup "TestGroup"...');
            SWIFT_StaffGroup::DeleteList([static::$_staffGroupId]);
        }
    }

    /**
     * Test POST endpoint
     * ** THIS SHOULD BE THE FIRST TEST IN ORDER TO TEST CRUD **
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function testPost(): void
    {
        $response = $this->getResponse('/Troubleshooter/Step', 'POST', [
            'subject' => 'Custom',
            'contents' => 'Test Contents',
            'categoryid' => static::$_categoryId,
            'staffid' => static::$_staffId,
        ]);
        $list = $this->getArrayFromResponse($response);
        $this->assertEquals('Custom', $list['troubleshooterstep']['subject']);
        static::$_id = $list['troubleshooterstep']['id'];
    }

    /**
     * Test GET all endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testPost')]
    public function testGetAll(): void
    {
        $response = $this->getResponse('/Troubleshooter/Step');
        $list = $this->getArrayFromResponse($response);
        $id = static::$_id;
        if (!isset($list['troubleshooterstep']['subject'])) {
            $list = array_values(array_filter($list['troubleshooterstep'], fn($ug) => $ug['id'] === $id));
        } else {
            $list = [$list['troubleshooterstep']];
        }
        $this->assertEquals('Custom', $list[0]['subject']);
    }

    /**
     * Test GET from ID endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testGetAll')]
    public function testGetFromId(): void
    {
        $response = $this->getResponse('/Troubleshooter/Step/' . static::$_id);
        $list = $this->getArrayFromResponse($response);
        $this->assertEquals('Custom', $list['troubleshooterstep']['subject']);
    }

    /**
     * Test PUT endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testGetFromId')]
    public function testPut(): void
    {
        $response = $this->getResponse('/Troubleshooter/Step/' . static::$_id, 'PUT', [
            'editedstaffid' => static::$_staffId,
            'subject' => 'NewCustom',
        ]);
        $list = $this->getArrayFromResponse($response);
        $this->assertEquals('NewCustom', $list['troubleshooterstep']['subject']);
    }

    /**
     * Test DELETE endpoint
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    #[\PHPUnit\Framework\Attributes\Depends('testPut')]
    public function testDelete(): void
    {
        $response = $this->getResponse('/Troubleshooter/Step/' . static::$_id, 'DELETE');
        $this->assertEmpty($response->getBody()->getContents());
    }
}
