<?php
/**
 * Copyright (c) 2013-2016
 *
 * @category  Library
 * @package   Dwoo\Block
 * @author    Jordi Boggiano <j.boggiano@seld.be>
 * @author    David Sanchez <david38sanchez@gmail.com>
 * @copyright 2008-2013 Jordi Boggiano
 * @copyright 2013-2016 David Sanchez
 * @license   http://dwoo.org/LICENSE Modified BSD License
 * @version   1.3.0
 * @date      2016-09-23
 * @link      http://dwoo.org/
 */

namespace Dwoo\Block;

use Dwoo\Plugin as DwooPlugin;
use Dwoo\Compiler as DwooCompiler;

/**
 * Base class for block plugins.
 * you have to implement the <em>init()</em> method, it will receive the parameters that
 * are in the template code and is called when the block starts
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from the use of this software.
 */
abstract class Plugin extends DwooPlugin
{
    /**
     * Stores the contents of the block while it runs.
     *
     * @var string
     */
    protected $buffer = '';

    /**
     * Buffers input, override only if necessary.
     *
     * @var string the content that must be buffered
     */
    public function buffer($input)
    {
        $this->buffer .= $input;
    }

    // initialization code, receives the parameters from {block param1 param2}
    // public function init($arg, $arg, ...);

    /**
     * Called when the block ends, this is most of the time followed right away by a call
     * of <em>process()</em> but not always, so this should be used to do any shutdown operations on the
     * block object, if required.
     */
    public function end()
    {
    }

    /**
     * Called when the block output is required by a parent block.
     * this must read $this->buffer and return it processed
     *
     * @return string
     */
    public function process()
    {
        return $this->buffer;
    }

    /**
     * Called at compile time to define what the block should output in the compiled template code, happens when the
     * block is declared basically this will replace the {block arg arg arg} tag in the template.
     *
     * @param DwooCompiler $compiler the compiler instance that calls this function
     * @param array        $params   an array containing original and compiled parameters
     * @param string       $prepend  that is just meant to allow a child class to call
     *                               parent::postProcessing($compiler, $params, "foo();") to add a command before the
     *                               default commands are executed parent::postProcessing($compiler, $params, "foo();")
     *                               to add a command before the default commands are executed
     * @param string       $append   that is just meant to allow a child class to call
     *                               parent::postProcessing($compiler, $params, null, "foo();") to add a command after
     *                               the default commands are executed parent::postProcessing($compiler, $params, null,
     *                               "foo();") to add a command after the default commands are executed
     * @param string       $type     the type is the plugin class name used
     *
     * @return string
     */
    public static function preProcessing(DwooCompiler $compiler, array $params, $prepend, $append, $type)
    {
        return DwooCompiler::PHP_OPEN . $prepend . '$this->addStack("' . $type . '", array(' . DwooCompiler::implode_r($compiler->getCompiledParams($params)) . '));' . $append . DwooCompiler::PHP_CLOSE;
    }

    /**
     * Called at compile time to define what the block should output in the compiled template code, happens when the
     * block is ended basically this will replace the {/block} tag in the template.
     *
     * @see preProcessing
     *
     * @param DwooCompiler $compiler the compiler instance that calls this function
     * @param array        $params   an array containing original and compiled parameters, see preProcessing() for more
     *                               details more details
     * @param string       $prepend  that is just meant to allow a child class to call
     *                               parent::postProcessing($compiler, $params, "foo();") to add a command before the
     *                               default commands are executed parent::postProcessing($compiler, $params, "foo();")
     *                               to add a command before the default commands are executed
     * @param string       $append   that is just meant to allow a child class to call
     *                               parent::postProcessing($compiler, $params, null, "foo();") to add a command after
     *                               the default commands are executed parent::postProcessing($compiler, $params, null,
     *                               "foo();") to add a command after the default commands are executed
     * @param string       $content  the entire content of the block being closed
     *
     * @return string
     */
    public static function postProcessing(DwooCompiler $compiler, array $params, $prepend, $append, $content)
    {
        return $content . DwooCompiler::PHP_OPEN . $prepend . '$this->delStack();' . $append . DwooCompiler::PHP_CLOSE;
    }
}
