<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Knowledgebase\Staff;

use Knowledgebase\Admin\LoaderMock;
use Knowledgebase\Models\Article\SWIFT_KnowledgebaseArticle;
use SWIFT;
use SWIFT_Exception;

/**
 * Class Controller_ArticleManagerTest
 */
#[\PHPUnit\Framework\Attributes\Group('knowledgebase')]
class Controller_ArticleManagerTest extends \SWIFT_TestCase
{
    public static $_record = [];

    public function setUp(): void
    {
        parent::setUp();

        unset($_POST);
    }

    /**
     * @return Controller_ArticleManagerMock
     * @throws SWIFT_Exception
     */
    public function getController(array $services = [])
    {
        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('AutoExecute')->willReturn(true);
        $mockDb->method('Insert_ID')->willReturn(1);
        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);
        $mockDb->method('QueryFetch')->willReturnCallback(function ($x) {
            if (str_contains($x, "kbarticleid = '0'")) {
                return [
                    'kbarticleid' => 0,
                ];
            }

            if (str_contains($x, "kbarticleid = '2'")) {
                return [
                    'kbarticleid' => 2,
                    'newstype' => '3',
                    'newsstatus' => '0',
                ];
            }

            return [
                'kbarticleid' => 1,
                'newstype' => '1',
                'newsstatus' => '2',
                'allowcomments' => '1',
                'uservisibilitycustom' => '1',
                'staffvisibilitycustom' => '2',
            ];
        });

        $this->mockProperty($mockDb, 'Record', [
            'kbarticleid' => '1',
            'kbcategoryid' => '1',
        ]);

        SWIFT::GetInstance()->Database = $mockDb;

        $mockStaff = $this->createMock(\Base\Models\Staff\SWIFT_Staff::class);

        $mockStaff->method('GetProperty')->willReturn(1);
        $mockStaff->method('GetStaffID')->willReturn(1);
        $mockStaff->method('GetIsClassLoaded')->willReturn(true);
        $mockStaff->method('GetPermission')
            ->willReturnOnConsecutiveCalls('0', '1');

        SWIFT::GetInstance()->Staff = $mockStaff;

        $mockSession = $this->createMock('SWIFT_Session');

        $mockSession->method('GetIsClassLoaded')->willReturn(true);
        $mockSession->method('GetProperty')->willReturnArgument(0);

        SWIFT::GetInstance()->Session = $mockSession;

        $mockRouter = $this->createMock('SWIFT_Router');

        SWIFT::GetInstance()->Router = $mockRouter;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturnOnConsecutiveCalls([], [
            [
                'appname' => 'news',
                'isenabled' => '1',
            ],
        ]);

        SWIFT::GetInstance()->Cache = $mockCache;

        $mockLang = $this->createMock('SWIFT_LanguageEngine');

        $mockLang->method('Get')->willReturnArgument(0);

        $mockView = $this->createMock('SWIFT_View');

        $mockInt = $this->createMock(\Base\Library\UserInterface\SWIFT_UserInterfaceControlPanel::class);

        $mockTpl = $this->createMock('SWIFT_TemplateEngine');

        $mockKb = $this->createMock(\Knowledgebase\Library\Article\SWIFT_KnowledgebaseArticleManager::class);
        $mockKb->method('RetrieveForStaff')->willReturnCallback(function ($x) {
            if ($x === 0) {
                return false;
            }
            if ($x === 2) {
                return [0];
            }
            return [
                new SWIFT_KnowledgebaseArticle(new \SWIFT_DataID(1)),
            ];
        });

        $mockMgr = $this->createMock(\Base\Library\Comment\SWIFT_CommentManager::class);

        $mockMgr->method('ProcessPOSTUser')->willReturn(true);

        $mockSettings = $this->createMock('SWIFT_Settings');

        $mockSettings->method('Get')->willReturn(1);

        return new Controller_ArticleManagerMock(array_merge([
            'Database' => $mockDb,
            'Template' => $mockTpl,
            'Language' => $mockLang,
            'UserInterface' => $mockInt,
            'View' => $mockView,
            'CommentManager' => $mockMgr,
            'Settings' => $mockSettings,
            'KnowledgebaseArticleManager' => $mockKb,
        ], $services));
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getController();
        $this->assertInstanceOf(\Knowledgebase\Staff\Controller_ArticleManager::class, $obj);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetLookupReturnsTrue(): never
    {
        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, true, false, true, false, true, false);
        $mockDb->Record = &self::$_record;

        $obj = $this->getController([
            'Database' => $mockDb,
        ]);
        $this->assertFalse($obj->GetLookup());

        $_POST['q'] = 'q';
        self::$_record = [
            'kbcategoryid' => 1,
            'parentkbcategoryid' => '0',
        ];
        $this->expectOutputRegex('/icon_kbarticle/');
        $this->assertTrue($obj->GetLookup());
        self::$_record['categorytype'] = 4;
        self::$_record['kbcategoryid'] = 2;
        $this->assertTrue($obj->GetLookup());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->GetLookup();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetMenuReturnsTrue()
    {
        $obj = $this->getController();
        $this->expectOutputRegex('/<ul>/');
        $this->assertTrue($obj->GetMenu());
        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->GetMenu();
    }

    /**
     * @throws SWIFT_Exception
     * @throws \ReflectionException
     */
    public function testRenderMenuReturnsHtml()
    {
        $obj = $this->getController();
        $mock = new \ReflectionClass($obj);
        $method = $mock->getMethod('RenderMenu');
        $method->setAccessible(true);
        $this->assertEquals('', $method->invoke($obj, []));
        $this->assertStringContainsString('<ul>', $method->invoke($obj, ['articles' => [1 => ['subject' => 'subject']]]));

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $method->invoke($obj, []);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetReturnsTrue()
    {
        $obj = $this->getController();
        $this->expectOutputRegex('/contentstext/');
        $this->assertFalse($obj->Get(0, 1));
        $this->expectOutputRegex('/contentstext/');
        $this->assertTrue($obj->Get(1, 1));
        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->Get(1, 1);
    }
}

class Controller_ArticleManagerMock extends Controller_ArticleManager
{
    /**
     * Controller_ArticleManagerMock constructor.
     * @param array $services
     * @throws SWIFT_Exception
     */
    public function __construct(array $services = [])
    {
        $this->Load = new LoaderMock();

        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }

        parent::__construct();
    }

    public function Initialize()
    {
        return true;
    }
}
