<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace News\Library\Render;

use News\Admin\LoaderMock;
use SWIFT;

/**
 * Class SWIFT_NewsRenderManagerTest
 */
#[\PHPUnit\Framework\Attributes\Group('news')]
class SWIFT_NewsRenderManagerTest extends \SWIFT_TestCase
{
    public static $_count = [];

    public function setUp(): void
    {
        parent::setUp();

        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);

        $this->mockProperty($mockDb, 'Record', [
            'newscategoryid' => 1,
            'totalitems' => 1,
        ]);

        SWIFT::GetInstance()->Database = $mockDb;

        $mockStaff = $this->createMock(\Base\Models\Staff\SWIFT_Staff::class);

        $mockStaff->method('GetProperty')->willReturnArgument(1);

        SWIFT::GetInstance()->Staff = $mockStaff;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturnOnConsecutiveCalls([
            '1' => [
                'newsitemcount' => '1',
            ]
        ]);

        SWIFT::GetInstance()->Cache = $mockCache;

        SWIFT::GetInstance()->Load = new LoaderMock();
    }

    public function getLibrary()
    {
        $mockLang = $this->createMock('SWIFT_LanguageEngine');

        $mockLang->method('Get')->willReturnCallback(function ($x) {
            if ($x === 'charset') {
                return 'UTF-8';
            }

            return $x;
        });
        \SWIFT::GetInstance()->Language = $mockLang;

        $mockSettings = $this->createMock('SWIFT_Settings');

        $mockSettings->method('Get')->willReturnCallback(function ($x) {
            if (!isset(self::$_count[$x])) {
                self::$_count[$x] = 0;
            }

            self::$_count[$x]++;

            if ($x === 'nw_enablestaffdashboard' && self::$_count[$x] > 1) {
                return '1';
            }

            return '0';
        });

        $mockTpl = $this->createMock('SWIFT_TemplateEngine');

        return new SWIFT_NewsRenderManagerMock([
            'Cache' => SWIFT::GetInstance()->Cache,
            'Language' => $mockLang,
            'Settings' => $mockSettings,
            'Template' => $mockTpl,
        ]);
    }

    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getLibrary();
        $this->assertInstanceOf(\News\Library\Render\SWIFT_NewsRenderManager::class, $obj);
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testRenderTreeReturnsHtml()
    {
        $obj = $this->getLibrary();
        $this->assertStringContainsString('<ul class="swifttree">', $obj->RenderTree());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->RenderTree();
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testRenderViewNewsTreeReturnsHtml()
    {
        $obj = $this->getLibrary();
        $this->assertStringContainsString('<ul class="swifttree">', $obj->RenderViewNewsTree());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->RenderViewNewsTree();
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testRenderWelcomeTabReturnsHtml()
    {
        $obj = $this->getLibrary();

        $this->assertEmpty($obj->RenderWelcomeTab(),
            'Returns empty string with nw_enablestaffdashboard = 0');

        $this->assertStringContainsString('containercontenttable', $obj->RenderWelcomeTab(),
            'Returns HTML for several items');

        $this->assertStringContainsString('<div class="dashboardmsg">', $obj->RenderWelcomeTab(),
            'Returns HTML for no items');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->RenderWelcomeTab();
    }
}

class SWIFT_NewsRenderManagerMock extends SWIFT_NewsRenderManager
{
    /**
     * SWIFT_NewsRenderManagerMock constructor.
     * @param array $services
     */
    public function __construct(array $services = [])
    {
        $this->Load = new LoaderMock();
        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }
        parent::__construct();
    }

    public function Initialize()
    {
        return true;
    }
}
