<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace Knowledgebase\Models\Article;

use Knowledgebase\Admin\LoaderMock;
use SWIFT;
use SWIFT_Data;
use SWIFT_DataID;
use SWIFT_DataStore;
use SWIFT_Exception;

/**
 * Class KnowledgebaseArticleLinkTest
 */
#[\PHPUnit\Framework\Attributes\Group('knowledgebase')]
class KnowledgebaseArticleLinkTest extends \SWIFT_TestCase
{
    /**
     * @param bool $loaded
     * @return LinkMock
     * @throws SWIFT_Exception
     */
    public function getModel($loaded = true, bool|array $pool = [1])
    {
        $mockCache = $this->getMockBuilder('SWIFT_CacheStore')
            ->disableOriginalConstructor()
            ->disableProxyingToOriginalMethods()
            ->getMock();

        $mockCache->method('Get')->willReturn([
            '1' => [
                'displayorder' => 0,
            ],
        ]);

        $mockDB = $this->createMock('SWIFT_Database');

        $mockDB->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);
        $mockDB->method('AutoExecute')->willReturn(true);
        $mockDB->method('QueryFetch')
            ->willReturn([
                'kbarticlelinkid' => 1,
                'linktype' => '1',
                'staffvisibilitycustom' => '1',
            ]);
        $mockDB->method('Insert_ID')
            ->willReturnOnConsecutiveCalls(1, 0);

        \SWIFT::GetInstance()->Database = $mockDB;
        \SWIFT::GetInstance()->Cache = $mockCache;

        $data = new \SWIFT_DataID(1);
        $data->SetIsClassLoaded($loaded);
        $obj = new LinkMock($data, $pool);
        $this->mockProperty($obj, 'Database', $mockDB);

        return $obj;
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testDestructCallsDestructor()
    {
        $obj = $this->getModel();
        $this->assertNotNull($obj);
        $obj->__destruct();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getModel();
        $this->assertInstanceOf(\Knowledgebase\Models\Article\SWIFT_KnowledgebaseArticleLink::class, $obj);
        $this->expectExceptionMessage('Failed to load Knowledgebase Article Link Object');
        $this->expectException('SWIFT_Exception');
        $this->getModel(false);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testProcessUpdatePoolReturnsFalse()
    {
        $obj = $this->getModel();
        $obj->SetUpdatePool([]);
        $this->assertFalse($obj->ProcessUpdatePool());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->ProcessUpdatePool();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetKnowledgebaseArticleLinkIdThrowsException()
    {
        $obj = $this->getModel();

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->GetKnowledgebaseArticleLinkID();
    }

    /**
     * @throws SWIFT_Exception
     * @throws \ReflectionException
     */
    public function testLoadDataThrowsException()
    {
        $obj = $this->getModel();
        $data = new SWIFT_DataStore([]);
        $data->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $mock = new \ReflectionClass($obj);
        $method = $mock->getMethod('LoadData');
        $method->setAccessible(true);
        $method->invoke($obj, $data);
    }

    /**
     * @throws SWIFT_Exception
     * @throws \ReflectionException
     */
    public function testLoadDataReturnsTrue()
    {
        $obj = $this->getModel();
        $data = new SWIFT_DataStore(['kbarticlelinkid' => 1]);
        $mock = new \ReflectionClass($obj);
        $method = $mock->getMethod('LoadData');
        $method->setAccessible(true);
        $this->assertTrue($method->invoke($obj, $data));
        $data = new SWIFT_DataStore([]);
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $method->invoke($obj, $data);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetPropertyThrowsException()
    {
        $obj = $this->getModel();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj->GetProperty('key');
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetPropertyReturnsValue()
    {
        $obj = $this->getModel();
        $this->assertEquals(1, $obj->GetProperty('kbarticlelinkid'));

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->GetProperty('key');
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testGetDataStoreReturnsArray()
    {
        $obj = $this->getModel();
        $this->assertIsArray($obj->GetDataStore());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->GetDataStore();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testCreateThrowsException()
    {
        $obj = $this->getModel();

        SWIFT::GetInstance()->Database->Insert_ID();
        $this->expectExceptionMessage(SWIFT_CREATEFAILED);
        $this->expectException('SWIFT_Exception');
        $obj::Create(1, 1, 1);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testCreateReturnsId()
    {
        $obj = $this->getModel();
        $this->assertEquals(1, $obj::Create(1, 1, 1));
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj::Create(0, 1, 1);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testDeleteListReturnsFalse()
    {
        $obj = $this->getModel();

        $this->assertFalse($obj::DeleteList([]));
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testDeleteReturnsTrue()
    {
        $obj = $this->getModel();
        $this->assertTrue($obj->Delete());

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $obj->Delete();
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testDeleteOnLinkTypeReturnsFalse()
    {
        $obj = $this->getModel();
        $this->assertFalse($obj::DeleteOnLinkType(1, []));
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj::DeleteOnLinkType(0, []);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testRetrieveLinkIdListOnArticleThrowsException()
    {
        $obj = $this->getModel();
        $this->expectExceptionMessage(SWIFT_INVALIDDATA);
        $this->expectException('SWIFT_Exception');
        $obj::RetrieveLinkIDListOnArticle(0, -1);
    }

    /**
     * @throws SWIFT_Exception
     */
    public function testDeleteOnKnowledgebaseArticleReturnsBoolean()
    {
        $obj = $this->getModel();

        $this->assertFalse($obj::DeleteOnKnowledgebaseArticle([]));
    }
}

class LinkMock extends SWIFT_KnowledgebaseArticleLink
{
    public function __construct(SWIFT_Data $_SWIFT_DataObject, private $_getPool = [1])
    {
        $this->Load = new LoaderMock();
        parent::__construct($_SWIFT_DataObject);
    }

    public function __destruct()
    {
        // prevent exception to be thrown when destroying the object and it's not loaded
        $this->SetIsClassLoaded(true);
        parent::__destruct();
    }

    public function GetUpdatePool()
    {
        return $this->_getPool;
    }

    public function SetUpdatePool($_pool)
    {
        return $this->_getPool = $_pool;
    }

    public function ProcessUpdatePool()
    {
        if (empty($this->_dataStore)) {
            return true;
        }
        return parent::ProcessUpdatePool();
    }
}