<?php
/**
 * ###############################################
 *
 * Kayako Classic
 * _______________________________________________
 *
 * @author        Werner Garcia <werner.garcia@crossover.com>
 *
 * @package       swift
 * @copyright     Copyright (c) 2001-2018, Trilogy
 * @license       http://kayako.com/license
 * @link          http://kayako.com
 *
 * ###############################################
 */

namespace News\Client;

use News\Admin\LoaderMock;
use SWIFT;

/**
 * Class Controller_NewsItemTest
 */
#[\PHPUnit\Framework\Attributes\Group('news')]
class Controller_NewsItemTest extends \SWIFT_TestCase
{
    public static $_cacheCount = [];

    public function setUp(): void
    {
        parent::setUp();

        $mockDb = $this->createMock('SWIFT_Database');

        $mockDb->method('AutoExecute')->willReturn(true);
        $mockDb->method('Insert_ID')->willReturn(1);
        $mockDb->method('NextRecord')->willReturnOnConsecutiveCalls(true, false);
        $mockDb->method('QueryFetch')->willReturnCallback(function ($x) {
            if (str_contains($x, "newsitemid = '2'")) {
                return false;
            }

            if (str_contains($x, "newsitemid = '3'")) {
                return [
                    'newsitemid' => 3,
                    'newstype' => '3',
                ];
            }

            return [
                'newsitemid' => 1,
                'newstype' => '1',
                'start' => '1',
                'expiry' => '2',
                'newsstatus' => '2',
                'allowcomments' => '1',
                'subject' => 'subject',
                'uservisibilitycustom' => '0',
            ];
        });

        $this->mockProperty($mockDb, 'Record', [
            'newsitemid' => '1',
        ]);

        SWIFT::GetInstance()->Database = $mockDb;

        $mockStaff = $this->createMock(\Base\Models\Staff\SWIFT_Staff::class);

        $mockStaff->method('GetProperty')->willReturnArgument(0);
        $mockStaff->method('GetStaffID')->willReturn(1);
        $mockStaff->method('GetIsClassLoaded')->willReturn(true);
        $mockStaff->method('GetPermission')
            ->willReturnOnConsecutiveCalls('0', '1');

        SWIFT::GetInstance()->Staff = $mockStaff;

        $mockSession = $this->createMock('SWIFT_Session');

        $mockSession->method('GetIsClassLoaded')->willReturn(true);
        $mockSession->method('GetProperty')->willReturnArgument(0);

        SWIFT::GetInstance()->Session = $mockSession;

        $mockRouter = $this->createMock('SWIFT_Router');

        SWIFT::GetInstance()->Router = $mockRouter;

        $mockCache = $this->createMock('SWIFT_CacheStore');

        $mockCache->method('Get')->willReturnCallback(function ($x) {
            if (!isset(self::$_cacheCount[$x])) {
                self::$_cacheCount[$x] = 0;
            }

            self::$_cacheCount[$x]++;

            if ($x === 'widgetcache' && self::$_cacheCount[$x] === 1) {
                return [
                    [
                        'appname' => 'news',
                        'isenabled' => '1',
                    ]
                ];
            }

            return [];
        });

        SWIFT::GetInstance()->Cache = $mockCache;
    }

    public function getController()
    {
        $mockLang = $this->createMock('SWIFT_LanguageEngine');

        $mockLang->method('Get')->willReturnArgument(0);

        $mockView = $this->createMock('SWIFT_View');

        $mockInt = $this->createMock(\Base\Library\UserInterface\SWIFT_UserInterfaceControlPanel::class);

        $mockTpl = $this->createMock('SWIFT_TemplateEngine');

        $mockMgr = $this->createMock(\Base\Library\Comment\SWIFT_CommentManager::class);

        $mockMgr->method('ProcessPOSTUser')->willReturn(true);

        return new Controller_NewsItemMock([
            'Template' => $mockTpl,
            'Language' => $mockLang,
            'UserInterface' => $mockInt,
            'View' => $mockView,
            'CommentManager' => $mockMgr,
        ]);
    }

    public function testConstructorReturnsClassInstance()
    {
        $obj = $this->getController();
        $this->assertInstanceOf(\News\Client\Controller_NewsItem::class, $obj);

        // covers app is installed
        $this->getController();
    }

    /**
     * @throws \SWIFT_Exception
     */
    public function testViewReturnsTrue()
    {
        $obj = $this->getController();

        $this->assertFalse($obj->View(0),
            'Returns false with invalid id');

        $this->assertFalse($obj->View(2),
            'Returns false with wrong id');

        $this->assertFalse($obj->View(3),
            'Returns false with newstype = private');

        $this->assertTrue($obj->View(1),
            'Returns true with correct id');

        $obj->SetIsClassLoaded(false);
        $this->expectExceptionMessage(SWIFT_CLASSNOTLOADED);
        $this->expectException('SWIFT_Exception');
        $this->assertFalse($obj->View(1));
    }

}

class Controller_NewsItemMock extends Controller_NewsItem
{
    /**
     * Controller_NewsItemMock constructor.
     * @param array $services
     */
    public function __construct(array $services = [])
    {
        $this->Load = new LoaderMock();

        foreach ($services as $prop => $service) {
            $this->$prop = $service;
        }

        parent::__construct();
    }

    public function Initialize()
    {
        return true;
    }

    protected function _ProcessNewsCategories() {
        // do nothing
    }
}
